package mcfall.raytracer.tests;

import java.awt.Dimension;

import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.math.Vector;

public class Camera extends TestCase {

	private mcfall.raytracer.Camera camera;
	private Point eye;
	private Point lookAt;	
	
	protected void setUp() throws Exception {
		eye = new Point (0, 1.0, 2.0);
		lookAt = new Point (0, 0, 0);
		camera = new mcfall.raytracer.Camera (eye, lookAt, 90, 1.5, 1, new Dimension (600, 400));
	}

	
	/**
	 * Values taken from example 7.2.1 in the Hill textbook
	 * eye = (4, 4, 4)
	 * look = (0, 1, 0)
	 * up = (0, 1, 0)
	 * Answers u=(4, 0, -4) (.7071, 0, -0.7071), v=(-12,32,-12) (-.3313, 0.8835, -.3313), n=(4,3,4) = (0.6247, 0.4685, 0.6247)
	 *
	 */
	public void testCameraAxes () {
		eye = new Point (4, 4, 4);
		lookAt = new Point (0, 1, 0);
		camera = new mcfall.raytracer.Camera(eye, lookAt, 90, 1, 1, new Dimension (100, 100));
		Vector u = camera.getU();
		Vector v = camera.getV();
		Vector n = camera.getN();
		assertAlmostEquals(0.7071, u.getValueAt(1));
		assertAlmostEquals(0.0, u.getValueAt(2));
		assertAlmostEquals(-0.7071, u.getValueAt(3));
		
		assertAlmostEquals(-0.3313, v.getValueAt(1));
		assertAlmostEquals(0.8835, v.getValueAt(2));
		assertAlmostEquals(-0.3313, v.getValueAt(3));
		
		assertAlmostEquals (0.6247, n.getValueAt(1));
		assertAlmostEquals (0.4685, n.getValueAt(2));
		assertAlmostEquals(0.6247, n.getValueAt(3));
							
	}
	/*
	 * Test method for 'mcfall.raytracer.Camera.rayThrough(int, int)'
	 */
	public void testRayThrough() {
		/*  
		 * Here's the camera situation:
		 * n = eye-look = (0, 1, 2) normalized gives (0.0, 0.44721, 0.89443)
		 * u = up x n = (2, 0, 0) normalized gives (1, 0, 0)
		 * v = n x u = (0, 4, -2) normalized gives (0, 0.8944, -0.4472)
		 * theta = 30
		 * aspect = 1.5
		 * N = 1
		 * H = N*tan(theta/2) = 1*tan(45) = 1
		 * W = H*aspect = 1.5
		 * nCols = 600
		 * nRows = 400
		 * 
		 * According to the book, then, the ray has starting point eye, and direction
		 * -Nn + W(2c/nCols-1)u+H(2r/nRows-1))v
		 * = (0, -0.44721, -0.89443) + 1.5(c/300-1)(1, 0, 0) + (r/200-1)(0, 0.8944, -0.4472)
		 */
		
		//  First check the camera axes
		Vector u = camera.getU();
		assertAlmostEquals(1.0, u.getValueAt(1));
		assertAlmostEquals(0.0, u.getValueAt(2));
		assertAlmostEquals(0.0, u.getValueAt(3));
		
		Vector v = camera.getV();
		assertAlmostEquals(0.0, v.getValueAt(1));
		assertAlmostEquals(0.8944, v.getValueAt(2));
		assertAlmostEquals(-0.4472, v.getValueAt(3));
		
		Vector n = camera.getN();
		
		
		//	  According to formula, we should get (0,-0.44721,-0.89443)+ -1.5(1,0,0) + -1(0,0.8944,-0.4472) 
		//  which is (-1.5, -1.3416, -0.4472)
		Ray throughUpperLeft = camera.rayThrough(0, 0);
		
		
		assertEquals (eye, throughUpperLeft.getStart());
		Vector direction = throughUpperLeft.getDirection();
		assertAlmostEquals (-1.5, direction.getValueAt(1));
		assertAlmostEquals (-1.3416, direction.getValueAt(2));
		assertAlmostEquals (-0.4472, direction.getValueAt(3));
		
		//  Try going through (1,1), which should give an answer of (-2.99, -4.98, -0.01)
		Ray throughOneOne = camera.rayThrough(1, 1);
		assertEquals (eye, throughOneOne.getStart());
		direction = throughOneOne.getDirection();
		assertAlmostEquals (-1.4950, direction.getValueAt(1));
		assertAlmostEquals (-1.3372, direction.getValueAt(2));
		assertAlmostEquals (-0.4495, direction.getValueAt(3));
		
		//  For the middle, we should get (0,-1,-2)+1.5*(0)(2,0,0)+0(0,4,-2)
		//  which is (0,-1,-2) + 0(2,0,0)= (0, -1, -2)
		//  This makes sense, because this is just -n
		Ray throughCenter = camera.rayThrough(300, 200);
		assertEquals (eye, throughCenter.getStart());
		direction = throughCenter.getDirection();
		assertAlmostEquals (0, direction.getValueAt(1));
		assertAlmostEquals (-0.4472, direction.getValueAt(2));
		assertAlmostEquals (-0.8944, direction.getValueAt(3));
		
		//  Finally try the bottom right corner, which should give (2.99, 2.98, -3.99)
		Ray throughBottomRight = camera.rayThrough(599, 399);
		assertEquals (eye, throughBottomRight.getStart());
		direction = throughBottomRight.getDirection();
		assertAlmostEquals (1.4950, direction.getValueAt(1));
		assertAlmostEquals (0.4427, direction.getValueAt(2));
		assertAlmostEquals (-1.3394, direction.getValueAt(3));
		
		
		//  Now, since this
	}

}
